﻿using Newtonsoft.Json.Linq;
using Tinycc.Exceptions;

namespace Tinycc;

public class Response
{
    public JObject Content { get; private set; }
    public bool IsSuccessStatusCode { get; private set; }
    public int StatusCode { get; private set; }

    public Response(bool isSuccessStatusCode, int statusCode, JObject content)
    {
        IsSuccessStatusCode = isSuccessStatusCode;
        StatusCode = statusCode;
        Content = content;

        // Check API error
        if (!IsSuccessStatusCode || GetApiErrorCode() != 0)
            throw new TinyccApiException(this);

        // Check API version
        var version = GetApiVersion();
        if (version != Constants.ApiVersion)
            throw new TinyccVersionException($"API version {version} is not supported, this client supports API version {Constants.ApiVersion}");
    }

    public T GetValue<T>(string name)
    {
        if (TryGetValue(name, out JObject result))
        {
            return result.ToObject<T>();
        }
        return default;
    }

    public T GetList<T>(string name)
    {
        if (TryGetValue(name, out JArray result))
        {
            return Content.ToObject<T>();
        }
        return default;
    }

    public Dictionary<string, T> GetDictionary<T>(string name)
    {
        if (TryGetValue(name, out JObject result))
        {
            Dictionary<string, T> dict = new Dictionary<string, T>();
            foreach (var c in result)
                dict.Add(c.Key, c.Value.ToObject<T>());
            return dict;
        }
        return default;
    }

    public string GetApiVersion()
    {
        if (!TryGetValue("version", out string version))
        {
            version = string.Empty;
        }

        return version;
    }

    public int GetApiErrorCode()
    {
        int errorCode = 0;
        if (TryGetValue("error", out JObject error))
            if (error.TryGetValue("code", out JToken code))
                errorCode = code.Value<int>();
        return errorCode;
    }

    public string GetApiErrorMessage()
    {
        string errorMessage = string.Empty;
        if (TryGetValue("error", out JObject error))
            if (error.TryGetValue("message", out JToken message))
                errorMessage = message.Value<string>() ?? string.Empty;
        return errorMessage;
    }

    protected bool TryGetValue<T>(string key, out T value)
    {
        JToken token;
        if (!Content.TryGetValue(key, StringComparison.OrdinalIgnoreCase, out token))
        {
            value = default(T);
            return false;
        }

        value = token.Value<T>();
        return true;
    }
}
